"""
Contains custom pygtk widgets for sudoku.

Copyright 2011 Bianca Gibson - bianca.rachel.gibson@gmail.com and 
Copyright 2005-2008, Thomas M. Hinkle

	This file is part of Sudoku.

	Sudoku is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Sudoku is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Sudoku.  If not, see <http://www.gnu.org/licenses/>.
"""
#KNOWN BUGS: 

import gtk, Sudoku, gobject, cairo, sudoku_console, pango, math

SPACING_FACTOR = 40 # amount of spacing between boxes in grid

BORDER_WIDTH = 9.0 # The size of space we leave for a box

BORDER_LINE_WIDTH = 4 # The size of the line we draw around a selected box

LITTLE_LINE_WIDTH = 0.25
NORMAL_LINE_WIDTH = 1

BASE_SIZE = 35 # The "normal" size of a box (in pixels)

SMALL_TO_BIG_FACTOR = 3 # The number of times wider than a small line a big line is.



class SudokuNumberBox(gtk.Widget):
	"""number box for the sudoku"""
	
	#set the text to the empty string by default
	text = ""
	possibilities_text = ""
	
	#set read only to false by default-for the boxes that values are known for
	read_only = False
	
	_layout = None
	_possibilities_layout = None
	text_color = None
	#highlight_color = ERROR_HIGHLIGHT_COLOR
	custom_background_color = None
	
	
	
	
	#set up the signals
	__gsignals__ = {
		'value-about-to-change':(gobject.SIGNAL_RUN_LAST,gobject.TYPE_NONE,()),
		'changed':(gobject.SIGNAL_RUN_LAST,gobject.TYPE_NONE,()),
		'undo-change':(gobject.SIGNAL_RUN_LAST,gobject.TYPE_NONE,()), 
		'notes-changed':(gobject.SIGNAL_RUN_LAST,gobject.TYPE_NONE,()),
	}
	
	#set state
	base_state = gtk.STATE_NORMAL
	draw_boxes = False
	
	
	def __init__(self, upper=9, text=''):
		"""initialise it"""
		gtk.Widget.__init__(self)		#init the super class
		self.upper = upper
		self.font = self.style.font_desc
		self.font.set_size(pango.SCALE*13)
		
		self.possibilities_font = self.font.copy()
		self.possibilities_font.set_size(pango.SCALE*7)
		
		#set properties-can focus and grab all events
		self.set_property('can-focus',True)
		self.set_property('events',gtk.gdk.ALL_EVENTS_MASK)
		
		#connect the events to functions
		self.connect('button-press-event',self.button_press)		#box is clicked on
		self.connect('key-release-event',self.key_press)			#release of key when inputting
		self.connect('enter-notify-event',self.pointer_enter)		#pointer enters
		self.connect('leave-notify-event',self.pointer_leave)		#pointer leaves
		self.connect('focus-in-event',self.focus_in)				#goes into focus
		self.connect('focus-out-event',self.focus_out)				#goes out of focus
		self.connect('motion-notify-event',self.motion_notify)		#poniter is in the window
		#print type(self)
		#cr = self.window.cairo_create()
		self.set_text(text)
		#self.do_realize()
		self.queue_draw()
		#self.cr = self.window.cairo_create()
		#print "made cr"
		
	#functions to set the state when appropriate
	def pointer_enter(self, *args):
		if not self.is_focus():
			self.set_state(gtk.STATE_PRELIGHT)
			
	def pointer_leave(self, *args):
		self.set_state(self.base_state)
		self._toggle_box_drawing_(False)
		
	def focus_in(self, *args):
		self.set_state(gtk.STATE_SELECTED)
		self.base_state = gtk.STATE_SELECTED
		
	def focus_out(self, *args):
		self.set_state(gtk.STATE_NORMAL)
		self.base_state = gtk.STATE_NORMAL
		
	
	def motion_notify(self, *args):
		"""activated when users mouse is in window. If the box is in focus will set drawing to true, otherwise false"""
		if self.is_focus():
			self._toggle_box_drawing_(True)
		else:
			self._toggle_box_drawing_(False)
			
	def _toggle_box_drawing_ (self, val):
		"""Sets self.draw_boxes to val"""
		self.draw_boxes = val
		self.queue_draw()
		
	def button_press(self, w, e):  
		"""W is the widget, e is the event."""
		if e.type == gtk.gdk._2BUTTON_PRESS:
			# ignore second click (this makes a double click in the
			# middle of a cell get us a display of the numbers, rather
			# than selecting a number.
			return
		elif self.is_focus():
			#get the coordinates of the click
			x,y = e.get_coords()        
			alloc = self.get_allocation()	#space allocation
			my_w = alloc.width		#width
			my_h = alloc.height		#height		
		else:
			self.grab_focus()
			
	def key_press(self, w, e):
		"""W is the widget, e is the event. Event for keypress"""
		#if the box is read only (one of the values read from the file
		#print 'key press'
		#print self.read_only
		if self.read_only:
			print 'read only'
			return
		
		#get the value that has been entered
		txt = gtk.gdk.keyval_name(e.keyval)
		if (txt == self.text):
			#no change, so return
			return
		if txt in ['0', 'Delete', 'Backspace']:
			self.set_text_interactive(txt)
		elif txt in ['1', '2', '3', '4', '5', '6', '7', '8', '9']:
			self.set_text_interactive(txt)
		
	
	def set_text(self, text):
		"""sets the text"""
		if text == "0":
			#print "is 0"
			self.text = ""
		else:
			self.text = text
		self._layout = self.create_pango_layout(self.text)
		self._layout.set_font_description(self.font)
		self.queue_draw()
	
	def set_text_interactive(self,text):
		"""Set the text interactively"""
		#emit the event, set the text, redraw and emit the changed event
		self.emit('value-about-to-change')
		self.set_text(text)
		self.emit('changed')
		self.queue_draw()
		
		
	def set_possibilities_text(self, possibilities_text=None):
		"""Sets the possibilities text to possibilities_text, if no value is given it is cleared"""
		self.possibilities_text = str(possibilities_text)
		self._possibilities_layout = self.create_pango_layout(self.possibilities_text)
		self._possibilities_layout.set_font_description(self.possibilities_font)
		self.queue_draw()
		
	def set_possibilities_text_interactive(self,text):
		"""Set the text interactively"""
		#emit the event, set the text, redraw and emit the changed event
		self.emit('value-about-to-change')
		self.set_possibilities_text(text)
		self.queue_draw()
		self.emit('changed') 
		
	def do_realize(self):
		"""Makes a window on which we draw"""
		
		#set a flag to say we are realised
		self.set_flags(self.flags() | gtk.REALIZED)
		
		#make a gdk window to draw on
		self.window = gtk.gdk.Window(self.get_parent_window(), width=self.allocation.width,
						height=self.allocation.height, window_type=gtk.gdk.WINDOW_CHILD,
						wclass=gtk.gdk.INPUT_OUTPUT, event_mask=self.get_events() | gtk.gdk.EXPOSURE_MASK)
		
		#associate the window with the widget
		self.window.set_user_data(self)
		
		#attach a style
		self.style.attach(self.window)
		
		#set the default background colour to whatever the theme says, and the state
		self.style.set_background(self.window, gtk.STATE_NORMAL)
		self.window.move_resize(*self.allocation)
		
	def do_unrealize(self):
		"""Unrealise the widget"""
		#removes the widget from the window
		self.window.set_user_data(None)
		
	def do_size_request(self, request):
		"""Used when GTK is asking the widget how much space it wants.
			No guarantee of getting all the space. """
		#ask for the text size plus a little border
		width, height = self._layout.get_size()
		request.width = (width / pango.SCALE)+(BORDER_LINE_WIDTH+BORDER_WIDTH)*2
		request.height = (height / pango.SCALE)+(BORDER_LINE_WIDTH+BORDER_WIDTH)*2
		
	def do_size_allocate(self, allocation):
		"""Called when the actual size is known and the widget finds out how much space it has"""
		#save the allocation
		self.allocation = allocation
		
		#if we are realised put the window at the location and size given
		if self.flags() & gtk.REALIZED:
			self.window.move_resize(*allocation)
			
	def do_expose_event(self, event):
		"""Called when the widget is asked to draw itself"""
		#put the allocation in some variables
		x, y, w, h = self.allocation
		cr = self.window.cairo_create()
		
		
		#if height is greater than width
		if (h > w):
			scale = h/float(BASE_SIZE)
		else:
			scale = w/float(BASE_SIZE)
		
		#set the scale
		cr.scale(scale,scale)
		
		#put in the colour
		self.draw_background_color(cr)
			
		#draw the highlight if necessary
		if self.is_focus():
			self.draw_highlight_box(cr)
		
		self.draw_normal_box(cr)
		self.draw_text(cr)
		if self.draw_boxes and self.is_focus():
			self.draw_note_area_highlight_box(cr)
			
	def draw_background_color(self, cr):
		"""fills in the background colour"""
		#if it's read only (one of the values given in the puzzle) make it insensitive colour
		#give it focused colour
		if self.read_only:
			cr.set_source_color(self.style.base[gtk.STATE_INSENSITIVE])
		elif self.is_focus():
			cr.set_source_color(self.style.base[gtk.STATE_SELECTED])
		elif (self.custom_background_color):
			cr.set_source_rgb(*self.custom_background_color)
		else:
			cr.set_source_color(self.style.base[self.state])
			
		cr.rectangle(0,0,BASE_SIZE,BASE_SIZE)
		cr.fill()
	   
	def set_to_default_colour(self, cr):
		cr.set_source_color(self.style.base[self.state])
		cr.rectangle(0,0,BASE_SIZE,BASE_SIZE)
		cr.fill()
		
	def draw_normal_box(self, cr):
		"""draw the box using cairo-pass in a cairo context"""
		#use cairo to draw the line around the box
		cr.set_source_color(self.style.mid[self.state])
		cr.rectangle(NORMAL_LINE_WIDTH*0.5, NORMAL_LINE_WIDTH*0.5,
			BASE_SIZE-NORMAL_LINE_WIDTH, BASE_SIZE-NORMAL_LINE_WIDTH)
		cr.set_line_width(NORMAL_LINE_WIDTH)
		cr.set_line_join(cairo.LINE_JOIN_ROUND)
		cr.stroke()
		
	def draw_highlight_box(self, cr):
		"""use cairo to draw the line around the highlighted box pass in a cairo context"""
		#set the colour, make the rectangle
		cr.set_source_color(self.style.base[gtk.STATE_SELECTED])
		cr.rectangle(BORDER_LINE_WIDTH*0.5, BORDER_LINE_WIDTH*0.5,
			BASE_SIZE-(BORDER_LINE_WIDTH), BASE_SIZE-(BORDER_LINE_WIDTH))
		
		#set the line width, join it up, and draw it
		cr.set_line_width(BORDER_LINE_WIDTH)
		cr.set_line_join(cairo.LINE_JOIN_ROUND)
		cr.stroke()
		
	def draw_note_area_highlight_box (self, cr):
		"""draw highlighted box with the note area up pass in a cairo context"""
		#set up the brush
		cr.set_source_color(self.style.mid[self.state])
		cr.set_line_width(NORMAL_LINE_WIDTH)
		cr.set_line_join(cairo.LINE_JOIN_ROUND)
		# make main rectangle
		self.draw_highlight_box(cr)
		
		# make the rectangle and draw it
		cr.rectangle(NORMAL_LINE_WIDTH*0.5, BASE_SIZE - BORDER_WIDTH-(NORMAL_LINE_WIDTH*0.5), 
					BASE_SIZE-NORMAL_LINE_WIDTH, BASE_SIZE-NORMAL_LINE_WIDTH)
		cr.stroke()
		
	def draw_text(self, cr):
		#get the text colour from the theme
		cr.set_source_color(self.style.text[self.state])
		
		#draw the text in the middle
		if self._layout:
			fontw, fonth = self._layout.get_pixel_size()
			cr.move_to((BASE_SIZE/2)-(fontw/2), (BASE_SIZE/2) - (fonth/2))
			cr.update_layout(self._layout)
			cr.show_layout(self._layout)
		
		if self._possibilities_layout:
			#draw the text around the bottum
			fontw, fonth = self._possibilities_layout.get_pixel_size()
			fonth = fonth
			cr.move_to(NORMAL_LINE_WIDTH, BASE_SIZE-fonth)
			cr.update_layout(self._possibilities_layout)
			cr.show_layout(self._possibilities_layout)
			
	def set_background_color(self, color):
		self.custom_background_color = color
		self.queue_draw()
		
	def set_value(self,v):
		if 0 < v <= self.upper:
			self.set_text(str(v))
		else:
			self.set_text('')
		self.queue_draw()
		
		
	def set_color(self, color):
		self.normal_color = color
		
		self.set_text_color(self.normal_color)

	def unset_color(self):
		self.set_color(None)

	def set_error_highlight(self, val):
		if val:
			self.set_text_color((1.0,0,0))
		else:
			self.set_text_color(self.normal_color)
		
	def set_read_only(self, val):
		self.read_only = val
#		if not hasattr(self,'bold_font'):
#			self.normal_font = self.font
#			self.bold_font = self.font.copy()
#			self.bold_font.set_weight(pango.WEIGHT_BOLD)
#		if self.read_only:
#			self.set_font(self.bold_font)
#		else:
#			self.set_font(self.normal_font)
#		self.queue_draw()

#not having it might have been the problem, CHECK WHAT THIS DOES		
gobject.type_register(SudokuNumberBox)


class SudokuNumberGrid(gtk.AspectFrame):
	"""Class for the grid of sudoku boxes"""
	
	def __init__(self, size):
		"""initialises the grid"""
		print ("init SudokuNumberGrid")
		#make a square table according to puzzle size, which defaults to 9. Homogeneos means all items in the table will be the same size.
		sudoku = Sudoku.Sudoku(size)
		self.size = size
		#print type(sudoku)
		self.table = gtk.Table(rows = sudoku.SIZE, columns = sudoku.SIZE, homogeneous=True)
		
		#store the puzzle size and initialise the entries to empty
		self.sudoku = sudoku
		self.__entries__ = [[0 for row in range(sudoku.SIZE)]for col in range(sudoku.SIZE)]
		
		#loop through the x and y coordinates of the grid
		for x in range(self.sudoku.SIZE):
			for y in range(self.sudoku.SIZE):
				#make a sudoku number box, set it's coordinates
				box = SudokuNumberBox(upper = self.sudoku.SIZE)
				#print type(box)
				box.x = x
				box.y = y
				
				#put it in the table and entries
				self.table.attach(box,x,x+1,y,y+1)
				self.__entries__[x][y] = box
				#print type(self.__entries__[x][y])
				
		#Do not obey child requests for aspect ratio, and set no shadow
		gtk.AspectFrame.__init__(self,obey_child=False)
		self.set_shadow_type(gtk.SHADOW_NONE)
		
		#add the table to self
		self.add(self.table)
		
		#set custom sizing method and show
		self.connect('size-allocate',self.allocate_size)
		self.show_all()
		
	def allocate_size(self, w, rect):
		"""handles the sizing of the grid"""
		
		if rect.width > rect.height: 
			side = rect.height
		else: 
			side = rect.width
			
		# calculate the spacing, based off SPACING_FACTOR constant
		spacing = int(rect.width / (self.size * SPACING_FACTOR))
		if spacing == 0: 
			spacing = 1
			
		self.change_spacing(spacing)
		
	def change_spacing (self, spacing):
		self.spacing = spacing
		self.big_spacing = spacing*SMALL_TO_BIG_FACTOR
		#self.table.set_row_spacings(spacing)
		#self.table.set_col_spacings(spacing)
		box_side = int(math.sqrt(self.size))
		#print box_side
		
		for n in range(1,box_side):
			self.table.set_row_spacing(box_side*n-1,self.big_spacing)
			self.table.set_col_spacing(box_side*n-1,self.big_spacing)
			self.table.set_border_width(self.big_spacing)
			
	def unhighlight_squares(self):
		for row in range(self.sudoku.SIZE):
			for column in range(self.sudoku.SIZE):
				self.__entries__[row][column].set_background_color(None)

		
